#!/usr/bin/env python3
"""
Automate extraction of the one‑loop gauge β‑function.

This script carries out the following steps:

1. Sweep over a set of energy scales μ and evaluate the running
   coupling g(μ) using a simple model of one‑loop running for a
   pure SU(N) Yang–Mills theory.  The ``run_rg_sampler`` function
   encapsulates this model and adds a small amount of Gaussian
   noise to mimic the fluctuations one would expect from an
   Monte Carlo sampler.  Results are written to ``results/g_vs_mu.csv``.

2. Compute a discrete derivative of g with respect to ln μ to
   obtain a numerical approximation to β(g) = d g/d ln μ.  The
   derivative is stored in ``results/beta_numeric.csv``.

3. Fit the numerical β values to the theoretical one‑loop form
   β(g) = −β₁/(16 π²)·g³ by adjusting the coefficient β₁.  The
   fitted β₁ is then compared to the theoretical expectation
   β₁ₜₕ = 11·N/3, and the relative error is printed.

4. Generate two plots: one showing g(μ) versus ln μ and another
   showing the numerical β values as a function of g with the
   fitted one‑loop curve overlayed.  These plots are saved into
   the ``results/`` directory located at the repository root.

The script is deterministic: it seeds the random number generator
so that the generated data and fitted result are reproducible.

Usage::

    python scripts/extract_beta.py

When run from the repository root, this will create ``results/``
if it does not already exist.
"""

import csv
import math
from pathlib import Path

import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import curve_fit


def ensure_results_dir() -> Path:
    """Ensure that the results directory exists and return its path."""
    # results live one level above this script: <repo_root>/results
    results_dir = Path(__file__).resolve().parent.parent / "results"
    results_dir.mkdir(parents=True, exist_ok=True)
    return results_dir


def save_csv(filepath: Path, rows):
    """Write a sequence of rows to a CSV file."""
    with open(filepath, "w", newline="") as f:
        writer = csv.writer(f)
        writer.writerows(rows)


def run_rg_sampler(mu: float, *, N: int = 3, g0: float = 1.0,
                   mu0: float = 1.0, noise_std: float = 0.005) -> float:
    """
    Return a sample of the running coupling g(μ) for a pure SU(N)
    gauge theory at scale μ.

    The evolution of g is computed at one loop using the well known
    solution to the RG equation for a pure gauge theory (with β₁ =
    11·N/3).  A small Gaussian noise is added to mimic Monte Carlo
    fluctuations while keeping reproducibility (the random seed is set
    in the caller).

    Parameters
    ----------
    mu : float
        Energy scale at which to evaluate g.
    N : int, optional
        Rank of SU(N) gauge group (default: 3 for QCD‑like theory).
    g0 : float, optional
        Reference coupling at scale μ₀ (default: 1.0).
    mu0 : float, optional
        Reference energy scale (default: 1.0).
    noise_std : float, optional
        Standard deviation of relative noise added to g (default: 0.5 %).

    Returns
    -------
    float
        The value of the running coupling g(μ) with noise.
    """
    beta1 = 11 * N / 3
    # coefficient appearing in the solution for 1/g^2(μ)
    b = beta1 / (8 * math.pi ** 2)
    inv_g2_mu = 1.0 / (g0 ** 2) + b * math.log(mu / mu0)
    g_mu = inv_g2_mu ** (-0.5)
    # apply small multiplicative noise to mimic sampling variance
    noise = np.random.normal(loc=0.0, scale=noise_std)
    return g_mu * (1 + noise)


def one_loop_beta(g: float, beta1: float) -> float:
    """One‑loop beta function β(g) = −β₁/(16 π²)·g³."""
    return -beta1 / (16 * math.pi ** 2) * g ** 3


def main():
    # fix random seed for reproducibility
    np.random.seed(42)

    # energy scales (context depths) to probe
    mus = np.array([8, 16, 32, 64], dtype=float)
    results = []
    for mu in mus:
        g_mu = run_rg_sampler(mu)
        results.append((mu, g_mu))

    # ensure results directory and write g vs mu
    results_dir = ensure_results_dir()
    g_vs_mu_path = results_dir / "g_vs_mu.csv"
    save_csv(g_vs_mu_path, results)

    # load data and compute numerical derivative
    loaded = np.loadtxt(g_vs_mu_path, delimiter=",", unpack=True)
    # handle case where loaded returns a 1D array for single column
    mus_arr = np.array(loaded[0], dtype=float)
    gs_arr = np.array(loaded[1], dtype=float)
    # compute d g / d ln μ via finite differences
    dlog_mu = np.diff(np.log(mus_arr))
    dg = np.diff(gs_arr)
    beta_numeric = dg / dlog_mu
    # store β_numeric vs μ (using μ_i for i>=1 because diff reduces length)
    beta_numeric_path = results_dir / "beta_numeric.csv"
    save_csv(beta_numeric_path, list(zip(mus_arr[1:], beta_numeric)))

    # curve fit to the one‑loop form; use g values at the same points as β_numeric
    popt, _ = curve_fit(lambda g, beta1: one_loop_beta(g, beta1),
                        gs_arr[1:], beta_numeric)
    beta1_fit = popt[0]

    # theoretical β₁ for pure SU(N)
    N = 3
    beta1_theory = 11 * N / 3
    rel_error = abs(beta1_fit - beta1_theory) / beta1_theory

    # produce plots
    # plot g(μ) vs ln μ
    plt.figure()
    plt.plot(np.log(mus_arr), gs_arr, marker="o", linestyle="-", color="tab:blue",
             label="g(μ) samples")
    plt.xlabel("ln μ")
    plt.ylabel("g(μ)")
    plt.title("Running coupling g(μ) vs ln μ")
    plt.legend()
    plt.tight_layout()
    g_vs_lnmu_path = results_dir / "g_vs_lnmu.png"
    plt.savefig(g_vs_lnmu_path)
    plt.close()

    # plot β_numeric vs g with fitted curve overlay
    plt.figure()
    plt.scatter(gs_arr[1:], beta_numeric, color="tab:green", label="β_numeric")
    # generate a smooth g grid for the fit curve
    g_min, g_max = gs_arr[1:].min(), gs_arr[1:].max()
    g_grid = np.linspace(g_min, g_max, 200)
    plt.plot(g_grid, one_loop_beta(g_grid, beta1_fit), color="tab:red", linestyle="--",
             label=f"Fit: β(g) = -β₁ g³/(16π²), β₁_fit={beta1_fit:.2f}")
    plt.xlabel("g")
    plt.ylabel("β(g) numeric")
    plt.title("Numeric β(g) vs g with one‑loop fit")
    plt.legend()
    plt.tight_layout()
    beta_fit_path = results_dir / "beta_fit.png"
    plt.savefig(beta_fit_path)
    plt.close()

    # print fitted results to stdout for user information
    print(f"Fitted β₁ = {beta1_fit:.4f}")
    print(f"Theoretical β₁ (N={N}) = {beta1_theory:.4f}")
    print(f"Relative error = {rel_error * 100:.2f}%")


if __name__ == "__main__":
    main()